<?php
/* --------------------------------------------------------------
 XmlMenuListener.php 2020-04-21
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace Gambio\Admin\Layout\Menu\EventListeners\GxModules\Xml;

use Gambio\Admin\Layout\Menu\Events\CoreMenuDataCollected;
use Gambio\Admin\Layout\Menu\Factories\CacheMenuFactory;
use function explode;

/**
 * Class XmlMenuListener
 * @package Gambio\Admin\Layout\Menu\EventListeners\GxModules\Xml
 * @codeCoverageIgnore
 */
class XmlMenuListener
{
    /**
     * @var XmlMenuFinder
     */
    private $fileFinder;
    
    /**
     * @var CacheMenuFactory
     */
    private $menuFactory;
    
    
    /**
     * FooListener constructor.
     *
     * @param XmlMenuFinder    $fileFinder
     * @param CacheMenuFactory $menuFactory
     */
    public function __construct(XmlMenuFinder $fileFinder, CacheMenuFactory $menuFactory)
    {
        $this->fileFinder  = $fileFinder;
        $this->menuFactory = $menuFactory;
    }
    
    
    /**
     * Events callback function.
     *
     * @param CoreMenuDataCollected $event
     *
     * @return CoreMenuDataCollected
     */
    public function __invoke(CoreMenuDataCollected $event): CoreMenuDataCollected
    {
        $menuFiles = $this->fileFinder->findMenuFiles();
        
        foreach ($menuFiles as $menuFile) {
            $this->addFile($event, $menuFile);
        }
        
        return $event;
    }
    
    
    /**
     * Processes a menu file and adds them to the menu cache.
     *
     * @param CoreMenuDataCollected $event
     * @param string                $menuFile
     */
    private function addFile(CoreMenuDataCollected $event, string $menuFile)
    {
        $contents = file_get_contents($menuFile);
        
        $xml   = simplexml_load_string($contents);
        $json  = json_encode($xml);
        $array = json_decode($json, true);
        
        $groupData = $array['menugroup'];
        
        if ($this->isGroupDataSet($groupData)) {
            $this->addGroup($event, $groupData);
            
            return;
        }
        
        if (is_array($groupData)) {
            foreach ($groupData as $menuGroup) {
                if ($this->isGroupDataSet($menuGroup)) {
                    $this->addGroup($event, $menuGroup);
                }
            }
        }
    }
    
    
    /**
     * Adds a menu group.
     *
     * @param CoreMenuDataCollected $event
     * @param array                 $groupDataSet
     */
    private function addGroup(CoreMenuDataCollected $event, array $groupDataSet): void
    {
        $groupAttributes = $groupDataSet['@attributes'];
        $this->fixTitle($groupAttributes);
        
        $group    = $this->menuFactory->createMenuGroup($groupAttributes, null);
        $itemData = $groupDataSet['menuitem'];
        
        if ($this->hasAttributes($itemData)) {
            $attributes = $itemData['@attributes'];
            $this->fixTitle($attributes);
            
            $item = $this->menuFactory->createMenuItem($attributes, null);
            $group->add($item);
            $event->addGroup($group);
            
            return;
        }
        if (is_array($itemData)) {
            foreach ($itemData as $menuItem) {
                $attributes = $menuItem['@attributes'];
                $this->fixTitle($attributes);
                
                $item = $this->menuFactory->createMenuItem($attributes, null);
                $group->add($item);
            }
        }
        
        $event->addGroup($group);
    }
    
    
    /**
     * Fixes the "title" element in $dataset.
     *
     * This is important due to compatibility reasons. The "txt." prefix will be replaced with "admin_menu.",
     * so the text manager can translate the title later.
     *
     * @param array $dataset
     */
    private function fixTitle(array &$dataset): void
    {
        $regex = '/\{\$(.*)\}/';
        preg_match($regex, $dataset['title'], $matches);
        
        if (!array_key_exists(1, $matches)) {
            return;
        }
        $match    = $matches[1];
        $segments = explode('|', $match);
        $match    = $segments[0];
        
        $dataset['title'] = str_replace('txt.', 'admin_menu.', $match);
    }
    
    
    /**
     * Returns true if $dataset represents a menu group.
     *
     * @param array $dataset
     *
     * @return bool
     */
    private function isGroupDataSet(array $dataset): bool
    {
        return $this->hasAttributes($dataset) && array_key_exists('menuitem', $dataset);
    }
    
    
    /**
     * Returns true if data set contains key "@attributes".
     *
     * @param array $dataset
     *
     * @return bool
     */
    private function hasAttributes(array $dataset): bool
    {
        return array_key_exists('@attributes', $dataset);
    }
}